/*
 * pi3's Linux kernel Runtime Guard
 *
 * Component:
 *  - Intercept 'generic_permission' function
 *
 * Notes:
 *  - Enforce Exploit Detection validation
 *
 * Caveats:
 *  - Originally, this file was placing hooks on 'may_open' function.
 *    Unfortunately, GCC (8+) might enable ISRA optimization when -Ox
 *    switch was used. During kernel compilation it is usually enabled,
 *    and as a side effect we have ISRA optimization as well
 *    (undesired for LKRG). ISRA performs interprocedural scalar
 *    replacement of aggregates, removal of unused parameters and
 *    replacement of parameters passed by reference by parameters passed
 *    by value. Since it's a very invasive modification ISRA changes
 *    symbol name of the functions which was modified.
 *    Alexander (Solar Designer) pointed out that in fact we could hook
 *    inode_permission() instead. This function is exported and
 *    additionally, we'll improve our coverage since it is called on more
 *    cases than 'may_open', including things such as permission checks
 *    on creating/removing of directories, (un)linking of files, and
 *    searching for files in directories with restricted permissions.
 *    LKRG hooks 'generic_permission' since this function is also exported
 *    and is called by inode_permission() after various checks have been
 *    made. It is also called in a few other specialized cases.
 *
 * Timeline:
 *  - Replace 'may_open' with 'generic_permission': 17.IX.2018
 *  - Created: 04.X.2017
 *
 * Author:
 *  - Adam 'pi3' Zabrocki (http://pi3.com.pl)
 *
 */

#include "../../../../p_lkrg_main.h"

static int p_generic_permission_entry(struct kprobe *p_ri, struct pt_regs *p_regs) {

   struct p_ed_process *p_tmp = NULL;

   p_ed_pcfi_cpu(1);

   if (p_is_ed_task(current)) {
      if ((p_tmp = ed_task_lock_current())) {
         p_ed_enforce_pcfi(NULL, p_tmp, p_regs);
         p_verify_addr_limit(p_tmp);
         p_ed_validate_current(p_tmp);
         ed_task_unlock(p_tmp);
      }
   }

   return 0;
}

static struct lkrg_probe p_generic_permission_probe = {
  .type = LKRG_KPROBE,
  .krp = {
    .kp.symbol_name = "generic_permission",
    .kp.pre_handler = p_generic_permission_entry,
  }
};

GENERATE_INSTALL_FUNC(generic_permission)
